% =========================================================================
%                光谱数据处理与可视化程序
% =========================================================================
%
% 功能描述:
%   1. 弹窗让用户选择光谱数据文件 (.txt, .csv, .xls, .xlsx等)。
%   2. 弹窗让用户选择数据类型：
%      a) 原始棍状光谱 (Raw Stick Spectrum)
%      b) 已展宽光谱 (Already Broadened Spectrum)
%   3. 如果是“原始数据”，则继续弹窗要求输入FWHM，进行高斯展宽，并
%      在一张图上对比显示原始谱和展宽谱。
%   4. 如果是“已展宽数据”，则直接读取并绘制光谱曲线。
%   5. 绘图时自动优化坐标轴范围。
%
% =========================================================================

%% 1. 初始化环境
% 清理工作区、命令窗口，关闭所有图形窗口
clear;         % 清除工作区中的所有变量
clc;           % 清空命令窗口
close all;     % 关闭所有已经打开的图形窗口

%% 2. 通过弹出式窗口获取数据文件
% 使用 uigetfile 让用户交互式地选择文件
[fileName, filePath] = uigetfile(...
    {'*.txt;*.csv;*.dat', '文本文件 (*.txt, *.csv, *.dat)'; ...
     '*.xls;*.xlsx', 'Excel 文件 (*.xls, *.xlsx)'; ...
     '*.*', '所有文件 (*.*)'}, ...
     '请选择光谱数据文件');

% 检查用户是否取消了文件选择
if fileName == 0
    disp('用户取消了操作。程序已终止。');
    return; % 如果取消，则退出程序
end

% 构建完整的文件路径
fullFilePath = fullfile(filePath, fileName);
fprintf('已选择文件: %s\n', fullFilePath);

%% 3. 读取文件数据
% 使用 try-catch 结构来处理可能发生的文件读取错误
try
    % readmatrix 是一个功能强大的函数，可以自动处理多种文件格式和分隔符
    rawData = readmatrix(fullFilePath);
    
    % 检查数据是否至少有两列
    if size(rawData, 2) < 2
        error('文件中的数据少于两列。请检查文件格式。');
    end
    
    % 提取X轴（波长）和Y轴（强度）数据
    % 此时我们尚不确定它是原始数据还是展宽数据，先统一命名
    data_x = rawData(:, 1); 
    data_y = rawData(:, 2);  
    
catch ME
    % 如果读取失败，则显示错误信息并终止程序
    errordlg(['读取文件时出错: ' ME.message], '文件读取错误');
    return;
end

%% 4. 新增：通过弹出式窗口让用户选择数据类型
% 使用 questdlg 创建一个带选项的对话框
question = '请选择您加载的数据类型：';
dlgTitle = '数据类型选择';
opt1 = '原始棍状光谱 (需要展宽)';
opt2 = '已展宽光谱 (直接绘图)';
opt3 = '取消';
defaultOption = opt1;
choice = questdlg(question, dlgTitle, opt1, opt2, opt3, defaultOption);

%% 5. 根据用户的选择，执行不同的操作
% 使用 switch 结构来处理不同的选择分支
switch choice
    % --- 分支A: 用户选择了“原始棍状光谱” ---
    case opt1
        fprintf('数据类型: 原始棍状光谱。即将进行高斯展宽。\n');
        
        % A1. 获取高斯展宽参数 (FWHM)
        prompt = {'请输入高斯展宽的半峰全宽 (FWHM):'};
        dlgTitle_fwhm = '设置展宽参数';
        defaultInput = {'10'};
        answer = inputdlg(prompt, dlgTitle_fwhm, 1, defaultInput);
        
        if isempty(answer)
            disp('用户取消了操作。程序已终止。');
            return;
        end
        
        fwhm = str2double(answer{1});
        if isnan(fwhm) || fwhm <= 0
            errordlg('输入的FWHM必须是一个正数。', '输入无效');
            return;
        end
        fprintf('高斯展宽FWHM设置为: %.2f\n', fwhm);

        % A2. 计算高斯展宽光谱
        wavelength_raw = data_x; % 明确变量名
        intensity_raw = data_y;
        
        margin = fwhm * 2;
        w_min = min(wavelength_raw) - margin;
        w_max = max(wavelength_raw) + margin;
        num_points = 2000;
        wavelength_broadened = linspace(w_min, w_max, num_points);
        
        sigma = fwhm / (2 * sqrt(2 * log(2)));
        intensity_broadened = zeros(size(wavelength_broadened));
        
        for i = 1:length(wavelength_raw)
            mu = wavelength_raw(i);
            A = intensity_raw(i);
            if A <= 0, continue; end
            gaussian_peak = A * exp(-(wavelength_broadened - mu).^2 / (2 * sigma^2));
            intensity_broadened = intensity_broadened + gaussian_peak;
        end
        disp('高斯展宽计算完成。');
        
        % A3. 绘图 (原始谱 + 展宽谱)
        figure;
        hold on;
        stem(wavelength_raw, intensity_raw, 'LineStyle', ':', 'Marker', 'none', 'Color', [0.7 0.7 0.7]);
        plot(wavelength_broadened, intensity_broadened, 'r-', 'LineWidth', 2);
        hold off;
        
        title(['光谱分析 (FWHM = ' num2str(fwhm) ')'], 'FontSize', 14);
        xlabel('波长 (Wavelength)', 'FontSize', 12);
        ylabel('能量/强度 (Intensity)', 'FontSize', 12);
        legend('原始棍状光谱', '高斯展宽光谱', 'Location', 'best');

    % --- 分支B: 用户选择了“已展宽光谱” ---
    case opt2
        fprintf('数据类型: 已展宽光谱。即将直接绘图。\n');
        
        % B1. 直接绘图
        figure;
        plot(data_x, data_y, 'b-', 'LineWidth', 2); % 使用蓝色实线绘图
        
        title('已展宽光谱图', 'FontSize', 14);
        xlabel('波长 (Wavelength)', 'FontSize', 12);
        ylabel('能量/强度 (Intensity)', 'FontSize', 12);
        % 对于单条曲线，通常不需要图例
        
    % --- 分支C: 用户选择了“取消”或关闭了对话框 ---
    case {opt3, ''}
        disp('用户取消了操作。程序已终止。');
        return;
end

% 对所有绘图情况都应用的通用设置
if ~isempty(choice) && ~strcmp(choice, opt3)
    % 只有在成功绘图后才执行以下代码
    axis tight; % 自适应坐标轴
    ax = gca;
    ax.YLim(2) = ax.YLim(2) * 1.1; % Y轴上限增加10%边距
    grid on;
    
    disp('光谱图绘制完成。');
end